///////////////
// Card List //
///////////////

#include "Card List.h"
#include "Card View.h"
#include "Filter.h"
#include "Library.h"
#include "Drawing.h"
#include "Dialogs.h"
#include "Basic IO.h"
#include <fstream.h>
#include <stdlib.h>
#include <stdio.h>
#include <strstrea.h>
#include <string.h>

static const int BOXES = 4;
static const int BOXWIDTH = BASE * 5;

static const int MAXKEY = 50;

void FreeCardList ();
BOOL LocateCardList (int, int);

void CardList::Create (const char *name, BOOL search, BOOL lock)
{
    const int TILES = 8;
    const int OFFSET = 32;
    static int Next = 0;
    RECT rect;
    int index, x, y;
    BOOL open;

    index = 0;
    open = FALSE;

    while ((index < TILES) && !open) {
        x = y = index * OFFSET;
        open = !LocateCardList (x, y);
        index ++;
    }

    if (!open) {
        x = y = Next * OFFSET;
        Next = ++Next % TILES;
    }

    rect.top = x;
    rect.left = y;
    rect.bottom = rect.top + (GetSystemMetrics (SM_CYSCREEN) / 2);
    rect.right = rect.left + WIDTHX + (GetSystemMetrics (SM_CXFRAME) * 2) + (GetSystemMetrics (SM_CXDLGFRAME) * 2) + GetSystemMetrics (SM_CXVSCROLL);

    Create (name, rect, search, lock, SORT_NAME);
}

void CardList::Create (const char *name, int x, int y, BOOL search, BOOL lock)
{
    const int OFFSET = 32;
    RECT rect;

    rect.left = x + OFFSET;
    rect.top = y + OFFSET;
    rect.right = rect.left + WIDTHX + (GetSystemMetrics (SM_CXFRAME) * 2) + (GetSystemMetrics (SM_CXDLGFRAME) * 2) + GetSystemMetrics (SM_CXVSCROLL);
    rect.bottom = rect.top + (GetSystemMetrics (SM_CYSCREEN) / 2);

    Create (name, rect, search, lock, SORT_NAME);
}

void CardList::Create (const char *name, const RECT& rect, BOOL search, BOOL lock, SORT sort)
{
    static epp_Class Class;
    static epp_Icon Icon;

    if (!Class) {
        Icon.LoadIcon (IDI_DECK);
        Class.Create (0, LoadCursor (NULL, IDC_ARROW), NULL, Icon);
    }

    Search = search;
    if (Search)
        _Locked = TRUE;
    _Sort = sort;
    _Find [0] = NULL;

    epp_Frame::Create (0, Class, NULL, WS_OVERLAPPEDWINDOW, rect, NULL);

    SetLock (lock);
    SetTitle (name);
    Views.Add (this);
    UpdateWindow ();
    UpdateStats ();
    UpdateBar ();
}

void CardList::SetTitle (const char *title)
{
    char temp [256];

    LoadString ((Search)? IDS_SEARCHTITLE : IDS_LISTTITLE, temp);
    strncat (temp, title, sizeof (temp) - strlen (temp) - 1);
    SetWindowText (temp);
    strncpy (Title, title, sizeof (Title) - 1);
}

void CardList::UpdateStats ()
{
    int index;
    int key;

    _Cards = 0;
    _Creatures = 0;
    _Spells = 0;
    _Artifacts = 0;
    _Lands = 0;
    _Points = 0;

    index = Cards.GetCount ();

    while (index --) {
        key = Cards.GetItemData (index);
        while (TRUE) {
            UpdateStats (key, 1);
            if (key & ~KEYMASK)
                key -= KEYINCR;
            else
                break;
        }
    }

    UpdateBar ();
}

char CardList::Sort ()
{
    if (Search)
        return ORDER_SEARCH;
    else
        return ORDER_DECK;
}

void CardList::Hide ()
{
    if (Search)
        Delete (NULL);
    else
        ShowWindow (SW_HIDE);
}

void CardList::Delete (HWND confirm)
{
    char message [256];

    sprintf (message, "Delete deck \"%s\"?", Title);

    if (!confirm || Search || (Cards.GetCount () == 0) || Check (message, confirm))
        Close ();
}

void CardList::Drop (int drop)
{
    UnlockDialog dialog;
    BOOL found;
    int index;
    int key;

    if (_Locked && !dialog.Run (*this))
        return;

    UpdateStats (drop, 1);
    UpdateBar ();

    found = FALSE;
    index = Cards.GetCount () - 1;
    drop &= KEYMASK;

    while (!found && (index >= 0)) {
        key = Cards.GetItemData (index);
        if (drop == (key & KEYMASK))
            found = TRUE;
        else
            index --;
    }

    if (found) {
        key += KEYINCR;
        Cards.SetItemData (index, key);
        if (Cards.GetCurSel () == index)
            InvalidateCard (index);
        else
            Cards.SetCurSel (index);
    } else
        Cards.SetCurSel (Cards.AddItemData (drop));
}

void CardList::Rename (HWND parent)
{
    RenameDialog dialog;

    if (dialog.Run (parent, Title, !Search)) {
        Search = !dialog.GetSave ();
        SetTitle (dialog.GetName ());
        Views.Refresh (this);
    }
}

void CardList::ModifyCard (int key)
{
    int index;
    int data;

    index = 0;

    while (index < Cards.GetCount ()) {
        data = Cards.GetItemData (index);
        if ((data & KEYMASK) == key) {
            Cards.DeleteString (index);
            Cards.AddItemData (data);
            break;
        } else
            index ++;
    }
}

void CardList::DeleteCard (int key)
{
    int index;

    index = 0;

    while (index < Cards.GetCount ())
        if ((Cards.GetItemData (index) & KEYMASK) == key) {
            Cards.DeleteString (index);
            break;
        } else
            index ++;
}

void CardList::Load (istream& file)
{
    char title [256];
    RECT rect;
    BOOL lock;
    SORT sort;

    SkipToDelim (file);
    ReadToDelim (file, title);
    ReadRect (file, rect);
    file >> lock;
    file >> (int) sort;

    Create (title, rect, FALSE, lock, sort);
}

void CardList::LoadData (istream& file)
{
    int key;

    while ((file >> key) && key)
        Cards.AddItemData (key);

    UpdateStats ();
}

void CardList::Save (ostream& file)
{
    if (Search) return;

    file << SAVED_DECK << ' ' << DELIM << Title << DELIM;
    WriteRect (file, _Rect);
    file << ' ' << _Locked << ' ' << (int) _Sort << endl;
}

void CardList::SaveData (ostream& file)
{
    int index;
    int key;

    if (Search) return;

    for (index = 0; index < Cards.GetCount (); index ++) {
        key = Cards.GetItemData (index);
        file << key << endl;
    }

    key = 0;
    file << key << endl;
}

HMENU CardList::GetMenu ()
{
    return ViewMenu;
}

int CardList::GetCard ()
{
    int index;

    index = Cards.GetCurSel ();

    if (index == LB_ERR)
        return 0;
    else
        return Cards.GetItemData (index);
}

HMENU CardList::GetContextMenu ()
{
    return ContextMenu;
}

void CardList::NewDeck ()
{
    char title [256];
    CardList *deck;

    LoadString (IDS_UNTITLED, title);
    deck = AllocCardList ();
    deck->Create (title, _Rect.left, _Rect.top, FALSE, FALSE);
    deck->Open ();
}

void CardList::NewCard ()
{
    int key;

    key = Views.NewCard (*this, FALSE);

    if (key) {
        Cards.SetCurSel (Cards.AddItemData (key));
        UpdateStats (key, 1);
        UpdateBar ();
    }
}

void CardList::Details ()
{
    CardView *card;
    int index;

    if (Cards.GetCount ()) {
        index = Cards.GetCurSel ();
        if (index != LB_ERR) {
            card = AllocCardView ();
            if (card)
                card->Create (Cards.GetItemData (index));
                card->Open ();
        }
    }
}

void CardList::ToggleLock ()
{
    SetLock (!_Locked);
}

void CardList::Add ()
{
    UnlockDialog dialog;
    int key;
    int index;

    if (Cards.GetCount ()) {
        index = Cards.GetCurSel ();
        if (index != LB_ERR) {
            if (_Locked && !dialog.Run (*this))
                return;
            key = Cards.GetItemData (index);
            UpdateStats (key, 1);
            UpdateBar ();
            Cards.SetItemData (index, key + KEYINCR);
            InvalidateCard (index);
        }
    }
}

static struct EXPORTDECK
{
    HWND ListBox;
    char Title [256];
    char Name [256];
    BOOL Direct;
};

static DWORD pascal ExportDeck (void *params)
{
    epp_ListBox listbox;
    ListFilter filter;
    CARD_INFO card;
    int index;
    int key;
    int count;

    EXPORTDECK *deck = (EXPORTDECK *) params;

    listbox.Attach (deck->ListBox);

    fstream file;
    strstream data;
    ostream *stream;

    if (deck->Direct) {
        stream = &data;
    } else {
        file.open (deck->Name, ios::in | ios::out | ios::trunc);
        if (!file) {
            Error ("Could not create export file", MB_ICONSTOP);
            delete deck;
            BusyEnd ();
            return 0;
        }
        stream = &file;
    }

    filter.WriteHeader (*stream, deck->Title);
    index = 0;

    while (*stream && (index < listbox.GetCount ())) {
        key = listbox.GetItemData (index++);
        count = (key >> 16) + 1;
        Library.Find (key, card);
        filter.Write (*stream, count, card.Name);
    }

    if (deck->Direct) {
        strstream redo;
        HANDLE block;
        unsigned size;
        char *dataptr;
        void *blockptr;
        char c;

        data.seekg (0);
        data.unsetf (ios::skipws);
        redo.unsetf (ios::skipws);

        while (data >> c) {
            if (c == '\n')
                redo << '\r';
            redo << c;
        }
        redo << ends;

        dataptr = redo.str ();
        size = strlen (dataptr) + 1;
        block = GlobalAlloc (GMEM_MOVEABLE | GMEM_DDESHARE, size);
        blockptr = GlobalLock (block);
        memcpy (blockptr, dataptr, size);
        redo.rdbuf ()->freeze (0);
        GlobalUnlock (block);

        if (OpenClipboard (listbox)) {
            SetClipboardData (CF_TEXT, block);
            CloseClipboard ();
        }
    }

    delete deck;
    BusyEnd ();
    return 0;
}

void CardList::Export (BOOL clipboard)
{
    ExportDeckDialog dialog;
    epp_Thread thread;
    EXPORTDECK *params;

    if (clipboard) {
        params = new EXPORTDECK;
        params->Direct = TRUE;
    } else {
        if (!dialog.Run (Title, *this))
            return;
        params = new EXPORTDECK;
        strcpy (params->Name, dialog);
        params->Direct = FALSE;
    }

    params->ListBox = Cards;
    strcpy (params->Title, Title);
    thread.Create (ExportDeck, params);
    BusyBegin (IDS_EXPORTDECK, *this);
}

void CardList::Remove ()
{
    UnlockDialog dialog;
    int index;
    int key;

    if (Cards.GetCount ()) {
        index = Cards.GetCurSel ();
        if (index != LB_ERR) {
            if (_Locked && !dialog.Run (*this))
                return;
            key = Cards.GetItemData (index);
            UpdateStats (key, -1);
            UpdateBar ();
            if (key & ~KEYMASK) {
                Cards.SetItemData (index, key - KEYINCR);
                InvalidateCard (index);
            } else
                Cards.DeleteString (index);
        }
    }
}

void CardList::FindCards ()
{
    SearchDialog dialog;
    CardList *deck;

    if (dialog.Run (*this, "Search Deck")) {
        deck = AllocCardList ();
        if (deck) {
            deck->Create (SearchTitle (dialog), _Rect.left, _Rect.top, TRUE, TRUE);
            deck->Open ();
            SearchLibrary (this, deck, dialog);
       }
    }
}

void CardList::FindCard ()
{
    FindDialog dialog;

    if (!Cards.GetCount ()) return;

    if (dialog.Run (*this, _Find) && strlen (dialog)) {
        strcpy (_Find, dialog);
        Cards.SetCurSel (LB_ERR);
        FindNext ();
    }
}

void CardList::FindNext ()
{
    CARD_INFO card;
    char mess [256];
    int items, index;
    int len, compare;
    int start;

    items = Cards.GetCount ();
    len = strlen (_Find);

    if (!items || !len) return;

    start = index = Cards.GetCurSel ();
    if (index == LB_ERR)
        index = 0;
    else {
        index ++;
        if (index == items)
          index = 0;
    }

    while (index < items) {
        Library.Find (Cards.GetItemData (index), card);
        if (memicmp (_Find, card.Name, len))
            index ++;
        else
            break;
    }

    if (index == items) {
        wsprintf (mess, "Card \"%s\" not found", _Find);
        Error (mess, MB_ICONEXCLAMATION);
        index = start;
    }

    if (index != LB_ERR)
        Cards.SetTopIndex (index);
    Cards.SetCurSel (index);
}

void CardList::SetChecks ()
{
    switch (_Sort) {
        case SORT_NAME:
            CheckMenuItem (Menu, IDM_DECK_SORTNAME, MF_BYCOMMAND | MF_CHECKED);
            CheckMenuItem (Menu, IDM_DECK_SORTTYPE, MF_BYCOMMAND | MF_UNCHECKED);
            break;
        case SORT_TYPE:
            CheckMenuItem (Menu, IDM_DECK_SORTNAME, MF_BYCOMMAND | MF_UNCHECKED);
            CheckMenuItem (Menu, IDM_DECK_SORTTYPE, MF_BYCOMMAND | MF_CHECKED);
            break;
    }
}

static DWORD pascal ResortCards (void *param)
{
    epp_ListBox cards;
    epp_Array <int> save;
    int items, index;

    cards.Attach ((HWND) param);
    items = cards.GetCount ();
    save.Alloc (items);

    for (index = 0; index < items; index ++)
        save [index] = cards.GetItemData (index);

    cards.ResetContent ();

    for (index = 0; index < items; index ++)
        cards.AddItemData (save [index]);

    BusyEnd ();
    return 0;
}

void CardList::SetSort (SORT sort)
{
    epp_Thread thread;

    if (_Sort == sort)
        return;

    _Sort = sort;
    SetChecks ();

    thread.Create (ResortCards, (HWND) Cards);
    BusyBegin (IDS_RESORTING, *this);
}

void CardList::SetLock (BOOL lock)
{
    _Locked = lock;
    Menu.CheckMenuItem (IDM_DECK_LOCK, MF_BYCOMMAND | (lock)? MF_CHECKED : MF_UNCHECKED);
}

void CardList::UpdateStats (int card, int incr)
{
    CARD_INFO info;
    BOOL creature;

    Library.Find (card, info);
    _Cards += incr;

    if (info.Abilities & LAND)
        _Lands += incr;
    else
        if (info.Abilities & CREATURE)
            _Creatures += incr;
        else
            _Spells += incr;

    if (info.Abilities & ARTIFACT)
        _Artifacts += incr;
}

void CardList::UpdateBar ()
{
    char text [256];
    int other;

    sprintf (text, "%i Card%s", _Cards, (_Cards == 1)? "" : "s");
    _StatusBar.SetText (text, 0, SBT_POPOUT);
    sprintf (text, "%i Beast%s", _Creatures, (_Creatures == 1)? "" : "s");
    _StatusBar.SetText (text, 1, SBT_POPOUT);
    other = _Cards - _Creatures - _Lands;
    sprintf (text, "%i Spell%s", other, (other == 1)? "" : "s");
    _StatusBar.SetText (text, 2, SBT_POPOUT);
    sprintf (text, "%i Land%s", _Lands, (_Lands == 1)? "" : "s");
    _StatusBar.SetText (text, 3, SBT_POPOUT);
}

void CardList::InvalidateCard (int index)
{
    RECT rect;
    Cards.GetItemRect (index, &rect);
    Cards.InvalidateRect (&rect);
}

void CardList::AdjustSize ()
{
    int edges [BOXES];
    int edge;
    RECT rect;
    RECT rect2;

    edge = 0;
    for (int box = 0; box < BOXES; box ++) {
        edge += BOXWIDTH;
        edges [box] = edge;
    }
    edges [BOXES - 1] = -1;
    _StatusBar.SetParts (BOXES, edges);

    GetClientRect (&rect);
    _ToolBar.GetClientRect (&rect2);
    rect.top += rect2.bottom + 1;
    _StatusBar.GetClientRect (&rect2);
    rect.bottom -= rect2.bottom;
    Cards.MoveWindow (rect);
}

static int CardNumber (const char *number)
{
    char temp [32];
    char *ptr, digit;

    ptr = temp;

    while (*number) {
        digit = *number++;
        if (strchr ("0123456789-", digit))
            *ptr++ = digit;
    }
    *ptr = NULL;

    return 255 - atoi (temp);
}

static int CardSortValue (CARD_INFO& card)
{
    int byte1, byte2, byte3, byte4;

    if (card.Abilities & LAND)
        byte1 = 8;
    else
        switch (card.Abilities & (RED | GREEN | WHITE | BLACK | BLUE)) {
            case RED:
                byte1 = 1;
                break;
            case GREEN:
                byte1 = 2;
                break;
            case WHITE:
                byte1 = 3;
                break;
            case BLACK:
                byte1 = 4;
                break;
            case BLUE:
                byte1 = 5;
                break;
            default:
                if (card.Abilities & ARTIFACT)
                    byte1 = 7;
                else
                    byte1 = 6;
                break;
        }

    if (card.Abilities & CREATURE)
        byte2 = 1;
    else
        byte2 = 2;

    byte3 = CardNumber (card.Power);
    byte4 = CardNumber (card.Tough);

    return (byte1 << 24) | (byte2 << 16) | (byte3 << 8) | byte4;
}

static void CardKey (char *key, CARD_INFO& card, SORT type)
{
    int len;

    switch (type) {
        case SORT_TYPE:
            sprintf (key, "%08x%s", CardSortValue (card), card.Name);
            break;
        case SORT_NAME:
        default:
            strcpy (key, card.Name);
            break;
    }
}

int CardList::OnCompareItem (int, COMPAREITEMSTRUCT *data)
{
    CARD_INFO card1, card2;
    char key1 [MAXKEY], key2 [MAXKEY];

    Library.Find (data->itemData1, card1);
    CardKey (key1, card1, _Sort);
    Library.Find (data->itemData2, card2);
    CardKey (key2, card2, _Sort);

    return strcmpi (key1, key2);
}

void CardList::OnCommand (int notify, int id, HWND)
{
    switch (id) {
        case IDM_DECK_LOAD:
            Views.ImportDeck (*this, FALSE);
            break;
        case IDM_DECK_LOADCLIP:
            Views.ImportDeck (*this, TRUE);
            break;
        case IDM_DECK_SAVE:
            Export (FALSE);
            break;
        case IDM_DECK_SAVECLIP:
            Export (TRUE);
            break;
        case IDM_DECK_PRINT:
            break;
        case IDM_DECK_NEW:
            NewDeck ();
            break;
        case IDM_DECK_RENAME:
            Rename (*this);
            break;
        case IDM_DECK_LOCK:
            ToggleLock ();
            break;
        case IDM_DECK_DELETE:
            Delete (*this);
            break;
        case IDM_DECK_NEWCARD:
            NewCard ();
            break;
        case IDM_DECK_OPEN:
        case IDM_DECKPOP_OPEN:
            Details ();
            break;
        case IDM_DECK_ADD:
        case IDM_DECKPOP_ADD:
            Add ();
            break;
        case IDM_DECK_REMOVE:
        case IDM_DECKPOP_REMOVE:
            Remove ();
            break;
        case IDM_DECK_SEARCH:
            FindCards ();
            break;
        case IDM_DECK_FIND:
            FindCard ();
            break;
        case IDM_DECK_FINDNEXT:
            if (strlen (_Find))
                FindNext ();
            else
                FindCard ();
            break;
        case IDM_DECK_SORTNAME:
            SetSort (SORT_NAME);
            break;
        case IDM_DECK_SORTTYPE:
            SetSort (SORT_TYPE);
            break;
        case IDM_DECK_MAIN:
            Views.ShowWindow (SW_NORMAL);
            Views.SetForegroundWindow ();
            break;
        case IDM_DECK_CLOSE:
            Hide ();
            break;
        case IDD_LIST:
            if (notify == LBN_DBLCLK)
              Details ();
            break;
    }
}

int CardList::OnCreate (CREATESTRUCT *)
{
    RECT rect;

	Menu.LoadMenu (IDR_DECK);
	ViewMenu.LoadMenu (IDR_MAINPOPDECK);
	ContextMenu.LoadMenu (IDR_DECKPOP);
	SetMenu (Menu);
    SetChecks ();

	const int COMMANDS [9] = { NULL, IDM_DECK_NEW, NULL, IDM_DECK_OPEN, IDM_DECK_FIND, IDM_DECK_FINDNEXT, NULL, IDM_DECK_ADD, IDM_DECK_REMOVE};
	_ToolBar.Create (*this, CCS_NOMOVEY | TBSTYLE_TOOLTIPS, IDB_DECKBAR, COMMANDS, 9);
	_ToolBar.ShowWindow ();
	_ToolBar.SetDelayTime (2500);
    _StatusBar.Create (WS_VISIBLE, *this);
    GetClientRect (&rect);
    Cards.Create (LBS_NOINTEGRALHEIGHT | LBS_OWNERDRAWFIXED | LBS_SORT | LBS_NOTIFY | LBS_DISABLENOSCROLL | WS_DLGFRAME | WS_VSCROLL | WS_VISIBLE, rect, *this, IDD_LIST);
    sCards.Create (Cards, this, this);
    sCards2.Create (Cards);

    AdjustSize ();
    Forward ();
    return 0;
}

void CardList::OnDestroy ()
{
    Views.Remove (this);
    FreeCardList ();
}

void CardList::OnDrawItem (int, DRAWITEMSTRUCT *data)
{
    CARD_INFO card;
    epp_Context dc;
    RECT rect;

    dc.Attach (data->hDC);

    if ((data->itemAction == ODA_DRAWENTIRE) || (data->itemAction == ODA_SELECT))
        if (Cards.GetCount ()) {
            Library.Find (data->itemData, card);
            DrawCardLine (dc, data->rcItem, card, (data->itemData >> 16) + 1, data->itemState & ODS_SELECTED);
        }
}

BOOL CardList::OnEraseBkgnd (HDC)
{
    return TRUE;
}

void CardList::OnGetMinMaxInfo (MINMAXINFO *info)
{
    static const int TOTALWIDTH = LINEWIDTH + (GetSystemMetrics (SM_CXFRAME) * 2) + (GetSystemMetrics (SM_CXDLGFRAME) * 2) + GetSystemMetrics (SM_CXVSCROLL);
    info->ptMaxSize.x = TOTALWIDTH;
    info->ptMaxTrackSize.x = TOTALWIDTH;
}

void CardList::OnMeasureItem (int, MEASUREITEMSTRUCT *data)
{
    data->itemWidth = LINEWIDTH;
    data->itemHeight = LINEHEIGHT;
}

void CardList::OnSetFocus (HWND)
{
    SetFocus (Cards);
}

void CardList::OnSize (int, int, int)
{
    _ToolBar.SendMessage (WM_SIZE, _wParam, _lParam);
    _StatusBar.SendMessage (WM_SIZE, _wParam, _lParam);
    AdjustSize ();
    Forward ();
}

BEGIN_HANDLER (CardList, ViewItem)
    ON_WM_DRAWITEM
    ON_WM_COMPAREITEM
    ON_WM_MEASUREITEM
    ON_WM_COMMAND
    ON_WM_SETFOCUS
    ON_WM_SIZE
    ON_WM_ERASEBKGND
    ON_WM_GETMINMAXINFO
    ON_WM_CREATE
    ON_WM_DESTROY
END_HANDLER

// allocator

static const int MAXLISTS = 500;
static int ListCount = 0;
static CardList CardLists [MAXLISTS];

CardList * AllocCardList ()
{
    int index;

    if (ListCount >= MAXLISTS)
        return NULL;

    index = 0;
    while (CardLists [index])
        index ++;

    ListCount ++;
    return &CardLists [index];
}

void FreeCardList ()
{
    ListCount --;
}

BOOL LocateCardList (int x, int y)
{
    RECT rect;
    int index;
    BOOL found;

    index = 0;
    found = FALSE;

    while ((index < MAXLISTS) && !found) {
        if (CardLists [index]) {
            rect = CardLists [index];
            if ((rect.left == x) && (rect.top == y))
                found = TRUE;
        }
        index ++;
    }

    return found;
}